<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\AndroidConfig;
use Kreait\Firebase\Messaging\CloudMessage;

class NotificationsController extends Controller {

   public function index(Request $request) {
      $notifications = Notification::orderBy('id', 'desc')->paginate(15);
      $count = Notification::count();

      $successMessage = "";
      if ($request->session()->pull('send-success', false)) {
         $successMessage = "Notification sent successfully";
      }

      return view('admin.notifications', [
         'notifications' => $notifications,
         'title' => "ALL NOTIFICATIONS ({$count})",
         'servers_json' => json_encode($notifications, JSON_UNESCAPED_SLASHES),
         'successMessage' => $successMessage,
      ]);
   }

   public function create(Request $request) {
      $notification = new Notification();
      $notification->topic = old('topic', "all");
      $notification->title = old('title', "");
      $notification->body = old('body', "");
      return view('admin.add-notification', [
         'title' => "SEND NOTIFICATION",
         'notification' => $notification,
      ]);
   }

   private function sendNotification(Notification $notification) {
      $factory = (new Factory())->withServiceAccount(storage_path('app/firebase-creds.json'));
      $messaging = $factory->createMessaging();
      $config = AndroidConfig::new()->withHighMessagePriority();
      $message = CloudMessage::withTarget('topic', $notification->topic)
         ->withData($notification->createFirebaseNotificationAsData())
         ->withAndroidConfig($config);
      $messaging->send($message);
   }

   public function saveAndSend(Request $request) {
      $request->validate([
         'topic' => 'required',
         'title' => 'required',
         'body' => 'required',
      ]);

      $notification = new Notification();
      $notification->topic = $request->input('topic');
      $notification->title = $request->input('title');
      $notification->body = $request->input('body');
      $notification->save();

      if (!App::environment("demo")) {
         $this->sendNotification($notification);
      }
      $request->session()->put('send-success', true);
      return redirect(route('notifications'));
   }

   public function duplicate(Request $request, $id) {
      $notification = Notification::findOrFail($id);

      if (!App::environment("demo")) {
         $this->sendNotification($notification);
      }

      $notification->id = null;
      $notification->exists = false;
      $notification->save();
      $request->session()->put('send-success', true);
      return response(true);
   }

   public function delete(Request $request, $id) {
      if ($id == 'debug' || $id = 'all') {
         Notification::where('topic', $id)->delete();
      } else {
         $server = Notification::findOrFail($id);
         $server->delete();
      }
      return response(true);
   }
}
