<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ConnectedDevice;
use App\Models\Server;
use App\Models\Setting;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rules\File as RulesFile;
use Illuminate\Validation\ValidationException;

class ServersController extends Controller {

    public function index(Request $request) {
        $servers = DB::select("SELECT s.* FROM servers AS s ORDER BY `order` ASC");

        $successMessage = "";
        if ($request->session()->pull('add-success', false)) {
            $successMessage = "Server added successfully";
        } elseif ($request->session()->pull('edit-success', false)) {
            $successMessage = "Server edited successfully";
        }

        header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
        header("Cache-Control: post-check=0, pre-check=0", false);
        header("Pragma: no-cache");

        return view('admin.servers', [
            'servers' => $servers,
            'title' => "SERVERS",
            'servers_json' => json_encode($servers, JSON_UNESCAPED_SLASHES),
            'successMessage' => $successMessage,
        ]);
    }

    public function add($showSuccess = false) {
        $server = new Server();
        $server->id = -1;
        $server->is_enabled = old('is_enabled', 1);
        $server->country_code = old('country_code', "us");
        $server->city = old('city');
        $server->ip = old('ip');
        $server->port = old('port', 1194);
        $server->vpn_username = old('vpn_username', "");
        $server->vpn_password = old('vpn_password', "");
        $server->free_connect_duration = old('free_connect_duration', 0);
        $server->capacity = old('capacity', 100);
        $server->protocol = old('protocol', "udp");
        $server->is_free = old('is_free', 1);
        $server->service_provider_link = old('service_provider_link', "");
        $server->use_file = old('use_file', false);
        $server->notes = old('notes', "");

        $links = DB::select("SELECT DISTINCT service_provider_link FROM servers WHERE service_provider_link!='' AND service_provider_link IS NOT NULL");

        return view('admin.add-server', [
            'action' => route('servers.store'),
            'title' => "NEW SERVER",
            'server' => $server,
            'links' => $links,
            'showSuccess' => $showSuccess,
        ]);
    }

    public function store(Request $request) {
        $request->validate([
            'country_code' => 'required|min:2|max:3',
            'port' => 'required|numeric|min:0|max:65535',
            'protocol' => 'required|in:tcp,udp',
            'capacity' => 'required|numeric|min:1',
            'free_connect_duration' => 'required|numeric|min:0',
            'service_provider_link' => 'nullable|url',
        ]);

        $allowDuplicateIPs = Setting::get('allow_duplicate_ips', '1');
        if ($allowDuplicateIPs == "1") {
            $request->validate(['ip' => 'required|ip']);
        } else {
            $request->validate(
                ['ip' => 'required|ip|unique:servers,ip'],
                ['ip.unique' => "Server with this IP does already exist."]
            );
        }

        $maxOrder = Server::max('order');

        $server = new Server();
        $server->is_enabled = $request->input('is_enabled') === 'on' ? 1 : 0;
        $server->country_code = strtolower($request->input('country_code'));
        $server->city = $request->input('city') ?? "";
        $server->ip = $request->input('ip');
        $server->port = $request->input('port', 1194);
        $server->vpn_username = $request->input('vpn_username') ?? "";
        $server->vpn_password = $request->input('vpn_password') ?? "";
        $server->order = $maxOrder ? $maxOrder + 1 : 0;
        $server->free_connect_duration = $request->input('free_connect_duration', 0);
        $server->capacity = $request->input('capacity', 100);
        $server->protocol = $request->input('protocol', 'udp');
        $server->is_free = $request->input('is_free', 1);
        $server->service_provider_link = $request->input('service_provider_link') ?? "";
        $server->notes = $request->input('notes', "");

        if ($request->hasFile('ovpn_file')) {
            self::validateOvpnFile($request, $server);
        } else {
            $server->use_file = false;
            $server->save();
        }

        if ($request->input('submit')) {
            return $this->add(true);
        }

        $request->session()->put('add-success', true);
        return redirect(route('servers.all'));
    }

    private static function validateOvpnFile(Request $request, Server $server) {
        $file = $request->file('ovpn_file');

        //Check extension and mime
        $allowedExtension = 'ovpn';
        $allowedMimeTypes = ['application/octet-stream', 'text/plain'];

        if (strtolower($file->getClientOriginalExtension()) !== $allowedExtension) {
            throw ValidationException::withMessages([
                'ovpn_file' => ['Invalid file extension, only .ovpn allowed'],
            ]);
        }

        if (!in_array($file->getMimeType(), $allowedMimeTypes)) {
            throw ValidationException::withMessages([
                'ovpn_file' => ['Invalid file type'],
            ]);
        }

        //Size check <= 1MB
        if ($file->getSize() > 1024 * 1024) {
            throw ValidationException::withMessages([
                'ovpn_file' => ['File size exceeds 1MB limit'],
            ]);
        }

        //Read content safely
        $content = file_get_contents($file->getRealPath());

        //Additional security checks: ensure it contains expected OVPN syntax
        if (!str_contains($content, 'client') || !str_contains($content, 'remote')) {
            throw ValidationException::withMessages([
                'ovpn_file' => ['Invalid OVPN configuration'],
            ]);
        }

        $server->use_file = true;
        //need to save server to get new id
        $server->save();
        if (!App::environment('demo')) {
            $file->storeAs('/public/uploaded_ovpn_files/', "{$server->id}.ovpn");
        }
    }

    public function download(Request $request) {
        $request->validate([
            'id' => 'required|numeric|min:0',
        ]);

        $id = $request->input('id');
        $type = $request->input('type', "created") == 'created' ? "created_ovpn_files" : "uploaded_ovpn_files";
        $file = storage_path("app/public/$type/") . "$id.ovpn";

        if (file_exists($file)) {
            return response()->download($file);
        }
        abort(404);
    }

    public function show($id) {
        $server = Server::findOrFail($id);
        $links = DB::select("SELECT DISTINCT service_provider_link FROM servers WHERE service_provider_link!='' AND service_provider_link IS NOT NULL");
        return view('admin.add-server', [
            'action' => route('servers.edit'),
            'title' => "SERVER INFO",
            'server' => $server,
            'links' => $links,
        ]);
    }

    public function edit(Request $request) {
        if ($request->input('use_file') == 1) {
            $request->validate([
                'country_code' => 'required|min:2|max:3',
                'capacity' => 'required|numeric|min:1',
                'free_connect_duration' => 'required|numeric|min:0',
                'service_provider_link' => 'nullable|url',
            ]);
        } else {
            $request->validate([
                'country_code' => 'required|min:2|max:3',
                'port' => 'required|numeric|min:0|max:65535',
                'protocol' => 'required|in:tcp,udp',
                'free_connect_duration' => 'required|numeric|min:0',
                'capacity' => 'required|numeric|min:1',
                'service_provider_link' => 'nullable|url',
            ]);
        }

        $server = new Server();
        $server->id = $request->input('id');
        $server->is_enabled = $request->input('is_enabled') === 'on' ? 1 : 0;
        $server->country_code = strtolower($request->input('country_code'));
        $server->city = $request->input('city') ?? "";
        $server->port = $request->input('port', 1194);
        $server->vpn_username = $request->input('vpn_username') ?? "";
        $server->vpn_password = $request->input('vpn_password') ?? "";
        $server->free_connect_duration = $request->input('free_connect_duration', 0);
        $server->capacity = $request->input('capacity', 100);
        $server->protocol = $request->input('protocol', 'udp');
        $server->is_free = $request->input('is_free', 1);
        $server->service_provider_link = $request->input('service_provider_link') ?? "";
        $server->notes = $request->input('notes', "");
        $server->exists = true;
        $server->save();

        if ($server->is_enabled == 0) {
            ConnectedDevice::where('server_id', $server->id)->update(['disconnect_time' => Carbon::now()]);
        }

        $request->session()->put('edit-success', true);
        return redirect(route('servers.all'));
    }

    public function reorder(Request $request) {
        $order = $request->input('order');
        for ($i = 0; $i < count($order); $i++) {
            $server = Server::where('id', $order[$i])->first();
            $server->order = $i;
            $server->save();
        }
        return response(true);
    }

    public function delete($id) {
        $server = Server::findOrFail($id);
        if ($server->use_file) {
            File::delete(storage_path("app/public/uploaded_ovpn_files/$id.ovpn"));
        }
        $server->delete();
        return response(true);
    }

    function resetConnectedDevices($id) {
        $server = Server::findOrFail($id);
        $server->connected_devices = 0;
        ConnectedDevice::where('server_id', $id)->delete();
        $server->save();
        return response(true);
    }

    public function ping(Request $request) {
        $ip = $request->input('ip');
        $status = ping_address($ip);
        return response()->json($status);
    }

    public function console($id) {
    }
}
